/*
 * Decompiled with CFR 0.152.
 */
package com.bc.ceres.util;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.TimeZone;
import java.util.stream.Stream;
import java.util.zip.GZIPOutputStream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class GenAUCatalog {
    private static final String NBM_FILE_EXTENSION = ".nbm";
    private static final String PATH_INFO_XML = "/Info/info.xml";
    private static final String FILE_NAME_CATALOG_XML = "updates.xml";
    private static final String FILE_NAME_CATALOG_XML_GZ = "updates.xml.gz";
    private static final String TAG_NAME_LICENSE = "license";
    private static final String TAG_NAME_MODULE = "module";
    private static final String ATTRIB_NAME_DOWNLOADSIZE = "downloadsize";
    private static final String ATTRIB_NAME_NAME = "name";
    private static final String XML_HEAD = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\r\n<!DOCTYPE module_updates PUBLIC \"-//NetBeans//DTD Autoupdate Catalog 2.5//EN\" \"http://www.netbeans.org/dtds/autoupdate-catalog-2_5.dtd\">\r\n\r\n";
    private final Path moduleDir;
    private final SimpleDateFormat timeStampFormat;
    private Transformer xmlTransformer;
    private DocumentBuilderFactory builderFactory;
    private Path catalogXmlPath;
    private Path catalogXmlGzPath;
    private String notificationMessage;
    private String notificationURL;

    public static void main(String[] args) throws Exception {
        if (args.length < 1) {
            GenAUCatalog.logErr("Please specify at least a path to a directory which contains NetBeans modules (*.nbm)");
        }
        Path moduleDir = Paths.get(args[0], new String[0]);
        GenAUCatalog generator = new GenAUCatalog(moduleDir);
        if (args.length > 1) {
            generator.setNotificationMessage(args[1]);
        }
        if (args.length > 2) {
            generator.setNotificationURL(args[2]);
        }
        generator.run();
    }

    public GenAUCatalog(Path moduleDir) throws Exception {
        this.moduleDir = moduleDir;
        this.catalogXmlPath = moduleDir.resolve(FILE_NAME_CATALOG_XML);
        this.catalogXmlGzPath = moduleDir.resolve(FILE_NAME_CATALOG_XML_GZ);
        if (Files.exists(this.catalogXmlPath, new LinkOption[0])) {
            throw new Exception(String.format("File %s already exists", this.catalogXmlPath));
        }
        if (Files.exists(this.catalogXmlGzPath, new LinkOption[0])) {
            throw new Exception(String.format("File %s already exists", this.catalogXmlGzPath));
        }
        this.timeStampFormat = new SimpleDateFormat("ss/mm/HH/dd/MM/yyyy");
        this.timeStampFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        this.builderFactory = DocumentBuilderFactory.newInstance();
        this.builderFactory.setValidating(false);
        this.builderFactory.setExpandEntityReferences(false);
        this.xmlTransformer = TransformerFactory.newInstance().newTransformer();
        this.xmlTransformer.setOutputProperty("omit-xml-declaration", "yes");
        this.xmlTransformer.setOutputProperty("indent", "yes");
    }

    public void setNotificationMessage(String msg) {
        this.notificationMessage = msg;
    }

    public void setNotificationURL(String url) {
        this.notificationURL = url;
    }

    private void run() throws Exception {
        ArrayList<Node> moduleList = new ArrayList<Node>();
        HashMap<String, Node> licenseMap = new HashMap<String, Node>();
        Stream<Path> nbmFiles = Files.list(this.moduleDir).filter(this::isNbmModuleFile);
        nbmFiles.forEach(path -> this.processModule((Path)path, moduleList, licenseMap));
        this.log(String.format("Found %d modules in directory", moduleList.size()));
        this.log(String.format("Modules have %d different licenses", licenseMap.size()));
        this.log("Writing: " + this.catalogXmlPath);
        this.writeCatalogXml(moduleList, licenseMap);
        this.log("Writing: " + this.catalogXmlGzPath);
        this.createGZipCatalog(this.catalogXmlPath, this.catalogXmlGzPath);
        String msg = "DONE";
        this.log(msg);
    }

    private void createGZipCatalog(Path catalogXmlPath, Path catalogXmlGzPath) throws IOException {
        try (InputStream inputStream = Files.newInputStream(catalogXmlPath, new OpenOption[0]);
             GZIPOutputStream gzOutStream = new GZIPOutputStream(Files.newOutputStream(catalogXmlGzPath, new OpenOption[0]));){
            int len;
            byte[] buf = new byte[51200];
            while ((len = inputStream.read(buf)) > 0) {
                gzOutStream.write(buf, 0, len);
            }
        }
    }

    private void writeCatalogXml(ArrayList<Node> moduleList, HashMap<String, Node> licenseMap) throws Exception {
        BufferedWriter writer = Files.newBufferedWriter(this.catalogXmlPath, new OpenOption[0]);
        writer.write(XML_HEAD);
        if (this.notificationMessage != null) {
            this.writeNotificationElement(writer);
        }
        writer.write(String.format("<module_updates timestamp=\"%s\">\r\n\r\n", this.timeStampFormat.format(Date.from(Instant.now()))));
        this.writeNodeCollection(moduleList, writer);
        this.writeNodeCollection(licenseMap.values(), writer);
        writer.write("</module_updates>\r\n");
        writer.close();
    }

    private void writeNotificationElement(BufferedWriter writer) throws IOException {
        String urlString = "";
        if (this.notificationURL != null) {
            urlString = String.format(" url=\"%s\"", this.notificationURL);
        }
        writer.write(String.format("<notification%s>%s</notification>\r\n\r\n", urlString, this.notificationMessage));
    }

    private boolean isNbmModuleFile(Path path) {
        return Files.isReadable(path) && path.getFileName().toString().endsWith(NBM_FILE_EXTENSION);
    }

    private void processModule(Path path, ArrayList<Node> moduleList, HashMap<String, Node> licenseMap) {
        try {
            this.log("Processing file: " + path.getFileName());
            FileSystem nbmFileSystem = FileSystems.newFileSystem(path, null);
            Path infoFile = nbmFileSystem.getPath(PATH_INFO_XML, new String[0]);
            DocumentBuilder documentBuilder = this.builderFactory.newDocumentBuilder();
            documentBuilder.setEntityResolver((publicId, systemId) -> new InputSource(new StringReader("")));
            Document document = documentBuilder.parse(Files.newInputStream(infoFile, new OpenOption[0]));
            Node moduleElement = document.getElementsByTagName(TAG_NAME_MODULE).item(0);
            GenAUCatalog.updateDownloadSizeAttribute(moduleElement, Files.size(path));
            moduleList.add(moduleElement);
            Node licenseElement = GenAUCatalog.getLicenseElement(moduleElement);
            if (licenseElement != null) {
                moduleElement.removeChild(licenseElement);
                this.addLicenseToMap(licenseElement, licenseMap);
            }
        }
        catch (IOException | ParserConfigurationException | SAXException e) {
            e.printStackTrace();
        }
    }

    private void addLicenseToMap(Node licenseElement, HashMap<String, Node> licenseMap) {
        NamedNodeMap attributes = licenseElement.getAttributes();
        if (attributes != null) {
            licenseMap.put(attributes.getNamedItem(ATTRIB_NAME_NAME).getNodeValue(), licenseElement);
        }
    }

    private void writeNodeCollection(Collection<Node> nodeCollection, BufferedWriter writer) throws Exception {
        for (Node node : nodeCollection) {
            StringWriter stringWriter = new StringWriter();
            this.xmlTransformer.transform(new DOMSource(node), new StreamResult(stringWriter));
            stringWriter.flush();
            writer.write(stringWriter.toString());
            writer.write("\r\n");
        }
    }

    private static void updateDownloadSizeAttribute(Node moduleElement, long sizeInBytes) {
        NamedNodeMap attributes = moduleElement.getAttributes();
        if (attributes == null) {
            return;
        }
        Node sizeItem = attributes.getNamedItem(ATTRIB_NAME_DOWNLOADSIZE);
        if (sizeItem == null) {
            return;
        }
        sizeItem.setNodeValue(String.valueOf(sizeInBytes));
    }

    private void log(String msg) {
        System.out.println(msg);
    }

    private static void logErr(String msg) {
        System.err.println(msg);
    }

    private static Node getLicenseElement(Node moduleElement) {
        NodeList childNodes = moduleElement.getChildNodes();
        for (int i = 0; i < childNodes.getLength(); ++i) {
            Node item = childNodes.item(i);
            if (!item.getNodeName().equals(TAG_NAME_LICENSE)) continue;
            return item;
        }
        return null;
    }
}

